<?php
/**
 * Proxy Payment Handler for Iran Server
 * این فایل باید روی سرور ایران (payesim.matinacard.ir) قرار بگیرد
 */

header('Content-Type: application/json; charset=utf-8');

// ZarinPal Configuration
define('ZARINPAL_MERCHANT_ID', 'fe6fc16e-0927-4dcd-95fe-07614d9f79ff');
define('ZARINPAL_API_URL', 'https://api.zarinpal.com/pg/v4/payment');

// Get request data
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data || !isset($data['action'])) {
    echo json_encode([
        'status' => 'error',
        'message' => 'درخواست نامعتبر'
    ]);
    exit;
}

// Handle different actions
switch ($data['action']) {
    case 'create_payment':
        handleCreatePayment($data);
        break;
    
    case 'verify_payment':
        handleVerifyPayment($data);
        break;
    
    default:
        echo json_encode([
            'status' => 'error',
            'message' => 'عملیات نامعتبر'
        ]);
}

/**
 * Create payment request with ZarinPal
 */
function handleCreatePayment($data) {
    $amount = intval($data['amount'] ?? 0);
    $description = $data['description'] ?? 'پرداخت eSIM';
    $callbackUrl = $data['callback_url'] ?? '';
    $email = $data['email'] ?? '';
    $mobile = $data['mobile'] ?? '';
    
    if ($amount < 1000) {
        echo json_encode([
            'status' => 'error',
            'message' => 'مبلغ نامعتبر است'
        ]);
        return;
    }
    
    if (empty($callbackUrl)) {
        echo json_encode([
            'status' => 'error',
            'message' => 'آدرس بازگشت تعریف نشده'
        ]);
        return;
    }
    
    // Prepare ZarinPal request
    $zarinpalData = [
        'merchant_id' => ZARINPAL_MERCHANT_ID,
        'amount' => $amount,
        'description' => $description,
        'callback_url' => $callbackUrl,
        'metadata' => [
            'email' => $email,
            'mobile' => $mobile
        ]
    ];
    
    // Make request to ZarinPal
    $ch = curl_init(ZARINPAL_API_URL . '/request.json');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($zarinpalData));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);
    
    if ($httpCode === 200 && $response) {
        $zarinpalResponse = json_decode($response, true);
        
        if ($zarinpalResponse && isset($zarinpalResponse['data']['code']) && $zarinpalResponse['data']['code'] == 100) {
            $authority = $zarinpalResponse['data']['authority'];
            $paymentUrl = 'https://www.zarinpal.com/pg/StartPay/' . $authority;
            
            echo json_encode([
                'status' => 'success',
                'authority' => $authority,
                'payment_url' => $paymentUrl
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'خطا در دریافت لینک پرداخت',
                'zarinpal_response' => $zarinpalResponse
            ]);
        }
    } else {
        echo json_encode([
            'status' => 'error',
            'message' => 'خطا در اتصال به زرین‌پال',
            'http_code' => $httpCode,
            'curl_error' => $curlError
        ]);
    }
}

/**
 * Verify payment with ZarinPal
 */
function handleVerifyPayment($data) {
    $authority = $data['authority'] ?? '';
    $amount = intval($data['amount'] ?? 0);
    
    if (empty($authority)) {
        echo json_encode([
            'status' => 'error',
            'message' => 'کد تراکنش ارسال نشده'
        ]);
        return;
    }
    
    if ($amount < 1000) {
        echo json_encode([
            'status' => 'error',
            'message' => 'مبلغ نامعتبر است'
        ]);
        return;
    }
    
    // Prepare verification request
    $zarinpalData = [
        'merchant_id' => ZARINPAL_MERCHANT_ID,
        'amount' => $amount,
        'authority' => $authority
    ];
    
    // Make request to ZarinPal
    $ch = curl_init(ZARINPAL_API_URL . '/verify.json');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($zarinpalData));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Content-Type: application/json',
        'Accept: application/json'
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode === 200 && $response) {
        $zarinpalResponse = json_decode($response, true);
        
        if ($zarinpalResponse && isset($zarinpalResponse['data']['code']) && $zarinpalResponse['data']['code'] == 100) {
            $refId = $zarinpalResponse['data']['ref_id'];
            
            echo json_encode([
                'status' => 'success',
                'ref_id' => $refId,
                'verified' => true
            ]);
        } elseif ($zarinpalResponse && isset($zarinpalResponse['data']['code']) && $zarinpalResponse['data']['code'] == 101) {
            // Already verified
            echo json_encode([
                'status' => 'success',
                'ref_id' => $zarinpalResponse['data']['ref_id'] ?? 'VERIFIED',
                'verified' => true,
                'message' => 'تراکنش قبلاً تایید شده'
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'تراکنش ناموفق یا لغو شده',
                'zarinpal_response' => $zarinpalResponse
            ]);
        }
    } else {
        echo json_encode([
            'status' => 'error',
            'message' => 'خطا در اتصال به زرین‌پال برای تایید',
            'http_code' => $httpCode
        ]);
    }
}

